
(function() {
  if (document.getElementById('code-jumper-host')) return;

  const DOMAIN = window.location.hostname;
  const STORAGE_KEY = `code_jumper_pos_${DOMAIN}`;

  const host = document.createElement('div');
  host.id = 'code-jumper-host';
  Object.assign(host.style, {
    position: 'fixed',
    zIndex: '2147483647',
    transition: 'opacity 0.4s cubic-bezier(0.4, 0, 0.2, 1)',
    opacity: '0',
    pointerEvents: 'none',
    left: '20px',
    bottom: '20px'
  });
  document.body.appendChild(host);

  const shadow = host.attachShadow({ mode: 'open' });
  const style = document.createElement('style');
  
  // Base styles with CSS variables for themes
  style.textContent = `
    :host { 
      --primary: #3b82f6; 
      --bg: #0f172a; 
      --border: #334155; 
      --text: #f8fafc; 
      --item-hover: #1e293b;
      --shadow: rgba(0,0,0,0.5);
    }
    
    :host(.light-mode) {
      --bg: #ffffff;
      --border: #e2e8f0;
      --text: #1e293b;
      --item-hover: #f1f5f9;
      --shadow: rgba(0,0,0,0.1);
    }

    .nav-container {
      background: var(--bg);
      border: 1px solid var(--border);
      border-radius: 24px;
      display: flex;
      flex-direction: column;
      padding: 6px;
      box-shadow: 0 10px 40px var(--shadow);
      width: 48px;
      height: 140px;
      box-sizing: border-box;
      align-items: center;
      justify-content: space-between;
      position: relative;
      pointer-events: auto;
      color: var(--text);
      transition: background 0.3s, border 0.3s, color 0.3s;
    }
    .drag-handle { 
      width: 20px; height: 3px; background: #94a3b8; border-radius: 10px; cursor: grab; margin-top: 4px; flex-shrink: 0;
    }
    button {
      background: transparent; border: none; color: var(--text); cursor: pointer;
      border-radius: 14px; display: flex; align-items: center; justify-content: center;
      transition: all 0.2s ease; width: 36px; height: 36px; font-size: 16px; z-index: 2;
    }
    button:hover { background: var(--item-hover); color: var(--primary); transform: translateY(-1px); }
    button:active { transform: translateY(1px) scale(0.95); }

    .count { 
      font-size: 10px; font-weight: 900; color: var(--primary); font-family: 'Inter', system-ui, sans-serif;
      cursor: pointer; width: 100%; text-align: center; height: 24px; display: flex; align-items: center; justify-content: center;
      z-index: 2; border-radius: 8px;
    }
    .count:hover { background: var(--item-hover); }
    
    .selection-menu {
      position: absolute; 
      left: 54px; 
      bottom: 0; 
      background: var(--bg); border: 1px solid var(--border);
      border-radius: 12px; box-shadow: 0 10px 30px var(--shadow);
      width: 32px;
      display: none; flex-direction: column; padding: 4px; z-index: 100;
      max-height: 200px; overflow-y: auto; scrollbar-width: none;
    }
    .selection-menu.open { display: flex; animation: slideIn 0.3s cubic-bezier(0.18, 0.89, 0.32, 1.28); }
    @keyframes slideIn { from { opacity: 0; transform: translateX(-10px) scale(0.9); } to { opacity: 1; transform: translateX(0) scale(1); } }
    
    .menu-item {
      width: 100%; height: 24px; text-align: center; font-size: 10px; font-weight: 800; cursor: pointer;
      border-radius: 6px; color: var(--text);
      margin-bottom: 2px; flex-shrink: 0;
      display: flex; align-items: center; justify-content: center;
      transition: all 0.1s;
    }
    .menu-item:hover { background: var(--item-hover); color: var(--primary); }
    .menu-item.active { background: var(--primary); color: white; }
  `;
  shadow.appendChild(style);

  const ui = document.createElement('div');
  ui.className = 'nav-container';
  ui.innerHTML = `
    <div class="drag-handle"></div>
    <button class="up" title="Previous Code Block (Ctrl+Up)">▲</button>
    <div class="count" title="Select Block">0/0</div>
    <button class="down" title="Next Code Block (Ctrl+Down)">▼</button>
    <div class="selection-menu"></div>
  `;
  shadow.appendChild(ui);

  const selectionMenu = ui.querySelector('.selection-menu');
  const countDisplay = ui.querySelector('.count');
  const dragHandle = ui.querySelector('.drag-handle');
  let blocks = [];
  let index = -1;

  // Smart Theme Detection Logic
  function updateTheme() {
    const isDarkMode = () => {
      // 1. Check system preference
      const systemDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
      
      // 2. Check body background color luminance
      const bgColor = window.getComputedStyle(document.body).backgroundColor;
      const rgb = bgColor.match(/\d+/g);
      if (rgb && rgb.length >= 3) {
        const luminance = (0.299 * rgb[0] + 0.587 * rgb[1] + 0.114 * rgb[2]);
        return luminance < 128; // If dark, return true
      }
      
      // 3. Fallback to common classes
      return document.documentElement.classList.contains('dark') || 
             document.body.classList.contains('dark-mode') || 
             systemDark;
    };

    if (isDarkMode()) {
      host.classList.remove('light-mode');
    } else {
      host.classList.add('light-mode');
    }
  }

  function updateVisibility() {
    if (blocks.length > 0) {
      host.style.opacity = '1';
      host.style.pointerEvents = 'auto';
    } else {
      host.style.opacity = '0';
      host.style.pointerEvents = 'none';
      selectionMenu.classList.remove('open');
    }
  }

  function findScrollableParent(node) {
    if (!node || node === document.body || node === document.documentElement) return window;
    const style = window.getComputedStyle(node);
    const isScrollable = (style.overflowY === 'auto' || style.overflowY === 'scroll') && node.scrollHeight > node.clientHeight;
    return isScrollable ? node : findScrollableParent(node.parentNode);
  }

  function scan() {
    updateTheme(); // Check theme on every scan to adapt to toggles
    const raw = document.querySelectorAll(`gm-code-block, code-block, .code-block-container, .markdown pre, pre, code-snippet, div[class*="code-block"], div[class*="syntax-highlighter"]`);
    const filtered = [];
    Array.from(raw).forEach(el => {
      if (el.innerText.trim().length < 5 || el.offsetWidth < 30) return;
      if (!filtered.some(ex => ex.contains(el))) filtered.push(el);
    });
    blocks = filtered;
    updateVisibility();
    if (blocks.length > 0) {
      if (index === -1) index = 0;
      index = Math.min(index, blocks.length - 1);
    } else index = -1;
    countDisplay.innerText = `${blocks.length > 0 ? index + 1 : 0}/${blocks.length}`;
    renderMenu();
  }

  function renderMenu() {
    selectionMenu.innerHTML = '';
    blocks.forEach((_, i) => {
      const item = document.createElement('div');
      item.className = `menu-item ${i === index ? 'active' : ''}`;
      item.innerText = i + 1;
      item.onclick = (e) => { e.stopPropagation(); jump(i); selectionMenu.classList.remove('open'); };
      selectionMenu.appendChild(item);
    });
  }

  function jump(newIndex) {
    scan();
    if (blocks.length === 0) return;
    if (newIndex < 0) newIndex = blocks.length - 1;
    if (newIndex >= blocks.length) newIndex = 0;
    index = newIndex;
    
    const element = blocks[index];
    const scrollContainer = findScrollableParent(element);
    const elementRect = element.getBoundingClientRect();
    const viewportHeight = window.innerHeight;
    
    if (scrollContainer === window) {
      const currentScroll = window.pageYOffset || document.documentElement.scrollTop;
      const elementAbsY = currentScroll + elementRect.top;
      window.scrollTo({
        top: elementAbsY - (viewportHeight * 0.4),
        behavior: 'smooth'
      });
    } else {
      const containerRect = scrollContainer.getBoundingClientRect();
      const relativeTop = elementRect.top - containerRect.top;
      scrollContainer.scrollTo({
        top: scrollContainer.scrollTop + relativeTop - (containerRect.height * 0.4),
        behavior: 'smooth'
      });
    }

    countDisplay.innerText = `${index + 1}/${blocks.length}`;
    renderMenu();
  }

  window.addEventListener('keydown', (e) => {
    if (e.ctrlKey && e.key === 'ArrowDown') { e.preventDefault(); jump(index + 1); }
    if (e.ctrlKey && e.key === 'ArrowUp') { e.preventDefault(); jump(index - 1); }
  });

  ui.querySelector('.up').onclick = (e) => { e.stopPropagation(); jump(index - 1); };
  ui.querySelector('.down').onclick = (e) => { e.stopPropagation(); jump(index + 1); };
  countDisplay.onclick = (e) => { e.stopPropagation(); selectionMenu.classList.toggle('open'); };
  
  dragHandle.addEventListener('mousedown', (e) => {
    let startX = e.clientX - host.offsetLeft;
    let startY = e.clientY - host.offsetTop;
    const move = (me) => { host.style.left = `${me.clientX - startX}px`; host.style.top = `${me.clientY - startY}px`; };
    const up = () => { 
      if (typeof chrome !== 'undefined' && chrome.storage) {
        chrome.storage.local.set({ [STORAGE_KEY]: { left: host.style.left, top: host.style.top } });
      }
      window.removeEventListener('mousemove', move); 
      window.removeEventListener('mouseup', up); 
    };
    window.addEventListener('mousemove', move);
    window.addEventListener('mouseup', up);
  });

  if (typeof chrome !== 'undefined' && chrome.storage) {
    chrome.storage.local.get([STORAGE_KEY], (result) => {
      if (result[STORAGE_KEY]) {
        const { left, top } = result[STORAGE_KEY];
        host.style.left = left; host.style.top = top;
      }
    });
  }

  window.addEventListener('click', () => selectionMenu.classList.remove('open'));
  new MutationObserver(() => scan()).observe(document.body, { childList: true, subtree: true });
  scan();
})();
